%Example for using Preconditioned matrix-oriented CG method with 
% rank truncation for solving matrix equation
% % A1 X B1' + ... + Am X Bm'  = R1_0 R2_0', 
%
%  The operator must be applied to P=P1*P2'  as 
%  Q1=OpMat1(P1) = [A1*P1, ...., Am*P1];
%  Q2=OpMat2(P2) = [B1*P2, ...., Bm*P2];
%
% The same for the preconditioners Z1=Prec1(P1)=[G1*P1, ...., Gm*P1], Z2=Prec2(P2)=...
%
% Final approximation:    X \approx  X1 *X2'


clear;

% Create data. Solve
%
%  A X + X A  + M X M = R01*R02'       
%
% equivalently, for X=X1*X2',   [A*X1 X1  M*X1][X2';X2'*A;X2'*M]  
%
% Operator applied at P=P1 * P2'  as  Q1*Q2' with Q1=[A*P1 P1 M*P1], Q2= [P2 A*P2 M'*P2]]
%
n=100;
ns=sqrt(n);
e = ones(ns,1);
T = spdiags([-e 2*e -e], -1:1, ns, ns);
I=speye(ns);
A = kron(I,T)+kron(T,I);

e = ones(n,1);
M=spdiags([-0.5*e -e 3.2*e -e -0.5*e], -2:2, n, n);
b=ones(n,1);b=b/norm(b);  
I=speye(n);



% Residual norm tolerance
tol=1e-6;

% truncation tolerance for CG factors. keep it small (e.g., 1e-14) if you do not want
% to activate truncation
tol_trunc=1e-10;
itmax=100;

% maximum allowed rank for iterates
rk1=n;rk2=n;

% zero starting guess
X1_0=0*b; X2_0=0*b;

OpMat1=@(P1)([ A*P1 P1 M*P1]);
OpMat2=@(P2)([ P2 A*P2 M'*P2]);

H1=ichol(A); Mp=ichol(M);
Prec1=@(P1)([ H1'\(H1\P1) P1 Mp'\(Mp\P1)]);
Prec2=@(P2)([P2 H1'\(H1\P2) Mp'\(Mp\P2)]);

% set mprint=0 if no residual plot
mprint=1;

tic
[X1,X2,res,it]=MotCG(OpMat1,OpMat2,Prec1,Prec2,X1_0,X1_0,b,b,rk1,rk2,tol,itmax,tol_trunc,mprint);
toc

Q1=OpMat1(X1);
Q2=OpMat2(X2);
fprintf('final true residual norm: %d\n', norm(b*b'-Q1*Q2'))
